# Correção: Bug de Saque Parcial - Sistema de Comissões

## 🐛 Problema Identificado

Quando um promoter solicitava um saque parcial (exemplo: R$3 de um saldo de R$10), o sistema marcava **TODAS** as comissões como 'solicitado' e posteriormente como 'pago', fazendo com que o valor total aparecesse como recebido ao invés de apenas o valor solicitado.

### Exemplo do Bug:
1. Promoter Carla tem R$10 disponíveis em comissões
2. Ela solicita saque de R$3
3. ❌ Sistema marcava os R$10 inteiros como 'solicitado'
4. Admin aprova o saque de R$3
5. ❌ Sistema marcava os R$10 como 'pago'
6. ❌ Carla via que recebeu R$10 ao invés de R$3
7. ❌ Carla perdia os R$7 restantes

## ✅ Solução Implementada

### 1. Nova Tabela: `saque_comissoes`

Criada uma tabela intermediária que rastreia exatamente quais comissões foram usadas em cada saque e quanto de cada uma foi utilizado.

**Arquivo:** `migration_fix_saque_parcial.sql`

```sql
CREATE TABLE IF NOT EXISTS saque_comissoes (
    id INT AUTO_INCREMENT PRIMARY KEY,
    saque_id INT NOT NULL,
    comissao_id INT NOT NULL,
    valor_usado DECIMAL(10,2) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (saque_id) REFERENCES saques(id) ON DELETE CASCADE,
    FOREIGN KEY (comissao_id) REFERENCES comissoes(id) ON DELETE CASCADE
);
```

### 2. Correções no `financeiro.php`

**Linhas modificadas:** 36-77 (saque PIX) e 94-137 (crédito de consumo)

**Nova lógica:**
- Busca comissões disponíveis ordenadas por data
- Marca apenas as comissões necessárias até cobrir o valor solicitado
- Registra na tabela `saque_comissoes` quais comissões foram usadas
- O restante das comissões permanece como 'disponivel'

### 3. Correções no `admin_saques.php`

**Rejeitar saque (linhas 37-80):**
- Devolve apenas as comissões relacionadas ao saque rejeitado
- Remove registros da tabela `saque_comissoes`

**Aprovar/Pagar saque (linhas 83-158):**
- Marca como 'pago' apenas as comissões relacionadas ao saque específico
- Usa a tabela `saque_comissoes` para identificar quais comissões pagar

## 📋 Como Aplicar a Correção

### Passo 1: Executar a Migration SQL

Execute o arquivo SQL no banco de dados:

```bash
mysql -u luminainkcom_promoter_user -p luminainkcom_promoters_system < migration_fix_saque_parcial.sql
```

Ou pelo phpMyAdmin:
1. Acesse phpMyAdmin
2. Selecione o banco `luminainkcom_promoters_system`
3. Vá em "SQL"
4. Copie e cole o conteúdo de `migration_fix_saque_parcial.sql`
5. Execute

### Passo 2: Substituir os Arquivos

Substitua os arquivos modificados:
- `financeiro.php`
- `admin_saques.php`

### Passo 3: Testar

1. **Teste de Saque Parcial:**
   - Crie um promoter com R$10 em comissões
   - Solicite saque de R$3
   - Verifique que:
     - ✅ Saldo disponível mostra R$7
     - ✅ Saldo solicitado mostra R$3
   - Admin aprova o saque
   - Verifique que:
     - ✅ Total recebido mostra R$3 (não R$10)
     - ✅ Saldo disponível ainda mostra R$7

2. **Teste de Múltiplos Eventos:**
   - Crie comissões de diferentes eventos
   - Solicite saque parcial
   - Verifique que todas as comissões são contabilizadas corretamente

## 🎯 Benefícios

✅ **Saques Parciais Corretos:** Promoters podem sacar valores parciais sem perder o restante

✅ **Rastreamento Preciso:** Sistema sabe exatamente quais comissões foram usadas em cada saque

✅ **Múltiplos Eventos:** Comissões de diferentes eventos são tratadas corretamente

✅ **Histórico Confiável:** O "Total Recebido" mostra o valor correto

✅ **Transações Seguras:** Uso de transações SQL previne inconsistências

## 📊 Fluxo Corrigido

```
Promoter com R$10 disponíveis
    ↓
Solicita saque de R$3
    ↓
Sistema marca apenas R$3 como 'solicitado'
    ↓
Saldo disponível: R$7
Saldo solicitado: R$3
    ↓
Admin aprova
    ↓
Marca apenas R$3 como 'pago'
    ↓
Total recebido: R$3 ✅
Saldo disponível: R$7 ✅
```

## 🔒 Compatibilidade

- ✅ Não afeta saques já processados
- ✅ Compatível com sistema de crédito de consumo
- ✅ Compatível com cartão de consumo
- ✅ Mantém todas as notificações funcionando
- ✅ Transações SQL garantem integridade dos dados

## 📝 Notas Técnicas

1. A tabela `saque_comissoes` usa `ON DELETE CASCADE` para manter integridade referencial
2. Transações SQL (`BEGIN TRANSACTION`, `COMMIT`, `ROLLBACK`) garantem que operações sejam atômicas
3. A lógica usa ordenação por `created_at ASC` para processar comissões mais antigas primeiro (FIFO)
4. O código é compatível com promoters participando de múltiplos eventos simultaneamente

## 🚀 Data da Correção

**09 de Dezembro de 2025**

---

**Desenvolvido por:** Claude AI
**Testado para:** Sistema PROMOTAAL6
