<?php
// admin_mensagens.php - Sistema de Mensagens Admin - Versão 3.0
require_once 'config.php';
verificarLogin();

// Verificar se é admin
if ($_SESSION['role'] !== 'admin') {
    header('Location: dashboard.php');
    exit;
}

$user_id = $_SESSION['user_id'];
$role = $_SESSION['role'];
$sucesso = '';
$erro = '';

// Buscar dados do usuário para sidebar
$stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
$stmt->execute([$user_id]);
$usuario = $stmt->fetch();

// Processar envio de mensagem
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['enviar_mensagem'])) {
    $assunto = filter_input(INPUT_POST, 'assunto', FILTER_SANITIZE_STRING);
    $mensagem = filter_input(INPUT_POST, 'mensagem', FILTER_SANITIZE_STRING);
    $destinatario_tipo = filter_input(INPUT_POST, 'destinatario_tipo', FILTER_SANITIZE_STRING);
    $evento_id = filter_input(INPUT_POST, 'evento_id', FILTER_SANITIZE_NUMBER_INT);
    $destinatario_individual = filter_input(INPUT_POST, 'destinatario_individual', FILTER_SANITIZE_NUMBER_INT);
    $prioridade = filter_input(INPUT_POST, 'prioridade', FILTER_SANITIZE_STRING) ?: 'normal';

    try {
        $pdo->beginTransaction();

        // Inserir mensagem
        $stmt = $pdo->prepare("
            INSERT INTO mensagens (remetente_id, remetente_nome, remetente_role, assunto, mensagem,
                                   tipo, destinatario_tipo, evento_id, prioridade)
            VALUES (?, ?, 'admin', ?, ?, ?, ?, ?, ?)
        ");

        $tipo = ($destinatario_tipo === 'individual') ? 'individual' :
                (($destinatario_tipo === 'evento_especifico') ? 'aviso_evento' : 'aviso_geral');

        $stmt->execute([
            $user_id,
            $usuario['nome'],
            $assunto,
            $mensagem,
            $tipo,
            $destinatario_tipo,
            $evento_id ?: null,
            $prioridade
        ]);

        $mensagem_id = $pdo->lastInsertId();

        // Definir destinatários
        $destinatarios = [];

        switch ($destinatario_tipo) {
            case 'todos':
                // Todos users e produtores
                $stmt = $pdo->query("SELECT id FROM users WHERE role IN ('user', 'produtor') AND status = 'aprovado'");
                $destinatarios = $stmt->fetchAll(PDO::FETCH_COLUMN);
                break;

            case 'promoters':
                // Apenas users (promoters)
                $stmt = $pdo->query("SELECT id FROM users WHERE role = 'user' AND status = 'aprovado'");
                $destinatarios = $stmt->fetchAll(PDO::FETCH_COLUMN);
                break;

            case 'produtores':
                // Apenas produtores
                $stmt = $pdo->query("SELECT id FROM users WHERE role = 'produtor' AND status = 'aprovado'");
                $destinatarios = $stmt->fetchAll(PDO::FETCH_COLUMN);
                break;

            case 'evento_ativo':
                // Users/Promoters com eventos ativos
                $stmt = $pdo->query("
                    SELECT DISTINCT c.promoter_id
                    FROM convidados c
                    INNER JOIN eventos e ON c.evento_id = e.id
                    INNER JOIN users u ON c.promoter_id = u.id
                    WHERE e.status = 'ativo'
                    AND c.promoter_id IS NOT NULL
                    AND u.status = 'aprovado'
                ");
                $destinatarios = $stmt->fetchAll(PDO::FETCH_COLUMN);
                break;

            case 'evento_especifico':
                // Users/Promoters de um evento específico
                if ($evento_id) {
                    $stmt = $pdo->prepare("
                        SELECT DISTINCT c.promoter_id
                        FROM convidados c
                        INNER JOIN users u ON c.promoter_id = u.id
                        WHERE c.evento_id = ?
                        AND c.promoter_id IS NOT NULL
                        AND u.status = 'aprovado'
                    ");
                    $stmt->execute([$evento_id]);
                    $destinatarios = $stmt->fetchAll(PDO::FETCH_COLUMN);
                }
                break;

            case 'individual':
                // Destinatário individual
                if ($destinatario_individual) {
                    $destinatarios = [$destinatario_individual];
                }
                break;
        }

        // Inserir destinatários
        if (!empty($destinatarios)) {
            $stmt = $pdo->prepare("
                INSERT INTO mensagens_destinatarios (mensagem_id, destinatario_id)
                VALUES (?, ?)
            ");

            foreach ($destinatarios as $dest_id) {
                $stmt->execute([$mensagem_id, $dest_id]);

                // Atualizar contador de mensagens não lidas
                $pdo->prepare("UPDATE users SET mensagens_nao_lidas = mensagens_nao_lidas + 1 WHERE id = ?")
                    ->execute([$dest_id]);

                // Criar notificação para o destinatário (Sistema v3.0)
                criarNotificacaoMensagem($dest_id, $mensagem_id, $assunto, $usuario['nome']);
            }
        }

        $pdo->commit();
        $sucesso = "Mensagem enviada com sucesso para " . count($destinatarios) . " destinatário(s)!";

    } catch (Exception $e) {
        $pdo->rollBack();
        $erro = "Erro ao enviar mensagem: " . $e->getMessage();
    }
}

// Buscar mensagens enviadas
$stmt = $pdo->prepare("
    SELECT m.*,
           COUNT(md.id) as total_destinatarios,
           SUM(CASE WHEN md.lida = 1 THEN 1 ELSE 0 END) as total_lidas
    FROM mensagens m
    LEFT JOIN mensagens_destinatarios md ON m.id = md.mensagem_id
    WHERE m.remetente_id = ?
    GROUP BY m.id
    ORDER BY m.created_at DESC
    LIMIT 50
");
$stmt->execute([$user_id]);
$mensagens_enviadas = $stmt->fetchAll();

// Buscar eventos para o select
$eventos = $pdo->query("SELECT id, nome FROM eventos WHERE status = 'ativo' ORDER BY nome")->fetchAll();

// Buscar usuários para mensagem individual
$usuarios = $pdo->query("
    SELECT id, nome, role FROM users
    WHERE role IN ('user', 'produtor') AND status = 'aprovado'
    ORDER BY nome
")->fetchAll();
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Mensagens - Admin</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="assets/dashboard-style.css">
    <link rel="stylesheet" href="assets/css/sidebar.css">
    <style>
        .container {
            max-width: 1400px;
            margin: 0 auto;
            padding: 30px 20px;
        }

        .page-header {
            margin-bottom: 30px;
        }

        .page-title {
            font-size: 32px;
            font-weight: 700;
            color: #FFD700;
            margin-bottom: 10px;
            display: flex;
            align-items: center;
            gap: 12px;
        }

        .page-subtitle {
            color: #999999;
            font-size: 16px;
        }

        .form-card {
            background: #1a1a1a;
            border: 2px solid #2a2a2a;
            border-radius: 15px;
            padding: 30px;
            margin-bottom: 30px;
        }

        .form-card:hover {
            border-color: #FFD700;
        }

        .form-section-title {
            font-size: 20px;
            font-weight: 700;
            color: #FFD700;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 2px solid #2a2a2a;
        }

        .form-row {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-bottom: 20px;
        }

        .form-group {
            display: flex;
            flex-direction: column;
            gap: 8px;
        }

        .form-label {
            font-size: 14px;
            font-weight: 600;
            color: #FFFFFF;
        }

        .form-input,
        .form-select,
        .form-textarea {
            width: 100%;
            padding: 12px;
            background: #0a0a0a;
            border: 2px solid #2a2a2a;
            border-radius: 10px;
            color: #FFFFFF;
            font-size: 14px;
            transition: all 0.3s;
        }

        .form-input:focus,
        .form-select:focus,
        .form-textarea:focus {
            outline: none;
            border-color: #FFD700;
        }

        .form-textarea {
            min-height: 120px;
            resize: vertical;
            font-family: inherit;
        }

        .priority-selector {
            display: grid;
            grid-template-columns: repeat(4, 1fr);
            gap: 10px;
        }

        .priority-option {
            position: relative;
        }

        .priority-option input[type="radio"] {
            display: none;
        }

        .priority-label {
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 12px;
            background: #0a0a0a;
            border: 2px solid #2a2a2a;
            border-radius: 10px;
            cursor: pointer;
            transition: all 0.3s;
            font-size: 13px;
            font-weight: 600;
        }

        .priority-option input[type="radio"]:checked + .priority-label {
            border-color: #FFD700;
            background: rgba(255, 215, 0, 0.1);
            color: #FFD700;
        }

        .priority-label.baixa { color: #999999; }
        .priority-label.normal { color: #FFD700; }
        .priority-label.alta { color: #FFA500; }
        .priority-label.urgente { color: #dc3545; }

        .btn-send {
            background: linear-gradient(135deg, #FFD700 0%, #FFA500 100%);
            color: #000;
            padding: 15px 30px;
            border: none;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 700;
            cursor: pointer;
            transition: all 0.3s;
            display: inline-flex;
            align-items: center;
            gap: 10px;
        }

        .btn-send:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 20px rgba(255, 215, 0, 0.4);
        }

        .messages-list {
            display: flex;
            flex-direction: column;
            gap: 15px;
        }

        .message-item {
            background: #1a1a1a;
            border: 2px solid #2a2a2a;
            border-radius: 15px;
            padding: 20px;
            transition: all 0.3s;
        }

        .message-item:hover {
            border-color: #FFD700;
            transform: translateX(5px);
        }

        .message-header {
            display: flex;
            justify-content: space-between;
            align-items: start;
            margin-bottom: 15px;
        }

        .message-subject {
            font-size: 18px;
            font-weight: 700;
            color: #FFD700;
            margin-bottom: 5px;
        }

        .message-meta {
            display: flex;
            gap: 15px;
            font-size: 13px;
            color: #999999;
        }

        .message-body {
            color: #CCCCCC;
            font-size: 14px;
            line-height: 1.6;
            margin-bottom: 15px;
        }

        .message-stats {
            display: flex;
            gap: 20px;
            padding-top: 15px;
            border-top: 1px solid #2a2a2a;
        }

        .stat-item {
            display: flex;
            align-items: center;
            gap: 8px;
            font-size: 13px;
        }

        .stat-icon {
            color: #FFD700;
        }

        .priority-badge {
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 11px;
            font-weight: 700;
            text-transform: uppercase;
        }

        .priority-badge.baixa {
            background: rgba(153, 153, 153, 0.2);
            color: #999999;
        }

        .priority-badge.normal {
            background: rgba(255, 215, 0, 0.2);
            color: #FFD700;
        }

        .priority-badge.alta {
            background: rgba(255, 165, 0, 0.2);
            color: #FFA500;
        }

        .priority-badge.urgente {
            background: rgba(220, 53, 69, 0.2);
            color: #dc3545;
        }

        @media (max-width: 768px) {
            .priority-selector {
                grid-template-columns: repeat(2, 1fr);
            }

            .form-row {
                grid-template-columns: 1fr;
            }

            .message-header {
                flex-direction: column;
                gap: 10px;
            }

            .message-stats {
                flex-wrap: wrap;
            }
        }
    </style>
</head>
<body>
    <?php include 'assets/sidebar.php'; ?>

    <div class="container">
        <div class="page-header">
            <h1 class="page-title">
                <i class="fas fa-envelope"></i>
                Enviar Mensagens e Avisos
            </h1>
            <p class="page-subtitle">Envie avisos para promoters e produtores do sistema</p>
        </div>

        <?php if ($sucesso): ?>
            <div class="alert alert-success">
                <i class="fas fa-check-circle"></i>
                <?= htmlspecialchars($sucesso) ?>
            </div>
        <?php endif; ?>

        <?php if ($erro): ?>
            <div class="alert alert-error">
                <i class="fas fa-exclamation-triangle"></i>
                <?= htmlspecialchars($erro) ?>
            </div>
        <?php endif; ?>

        <!-- Formulário de Envio -->
        <form method="POST" class="form-card">
            <h2 class="form-section-title">
                <i class="fas fa-paper-plane"></i>
                Nova Mensagem
            </h2>

            <div class="form-row">
                <div class="form-group">
                    <label class="form-label">Assunto *</label>
                    <input type="text" name="assunto" class="form-input" required
                           placeholder="Ex: Novo evento disponível">
                </div>

                <div class="form-group">
                    <label class="form-label">Destinatários *</label>
                    <select name="destinatario_tipo" id="destinatario_tipo" class="form-select" required>
                        <option value="">Selecione...</option>
                        <option value="todos">Todos os Promoters e Produtores</option>
                        <option value="promoters">Apenas Promoters</option>
                        <option value="produtores">Apenas Produtores</option>
                        <option value="evento_ativo">Promoters com Eventos Ativos</option>
                        <option value="evento_especifico">Promoters de um Evento Específico</option>
                        <option value="individual">Usuário Individual</option>
                    </select>
                </div>
            </div>

            <!-- Campo de Evento (aparece quando seleciona evento específico) -->
            <div class="form-row" id="campo_evento" style="display: none;">
                <div class="form-group">
                    <label class="form-label">Selecionar Evento *</label>
                    <select name="evento_id" class="form-select">
                        <option value="">Escolha o evento...</option>
                        <?php foreach ($eventos as $evento): ?>
                            <option value="<?= $evento['id'] ?>">
                                <?= htmlspecialchars($evento['nome']) ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>

            <!-- Campo de Usuário Individual -->
            <div class="form-row" id="campo_individual" style="display: none;">
                <div class="form-group">
                    <label class="form-label">Selecionar Usuário *</label>
                    <select name="destinatario_individual" class="form-select">
                        <option value="">Escolha o usuário...</option>
                        <?php foreach ($usuarios as $user): ?>
                            <option value="<?= $user['id'] ?>">
                                <?= htmlspecialchars($user['nome']) ?>
                                (<?= $user['role'] === 'user' ? 'Promoter' : 'Produtor' ?>)
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>

            <div class="form-group">
                <label class="form-label">Mensagem *</label>
                <textarea name="mensagem" class="form-textarea" required
                          placeholder="Digite sua mensagem aqui..."></textarea>
            </div>

            <div class="form-group">
                <label class="form-label">Prioridade</label>
                <div class="priority-selector">
                    <div class="priority-option">
                        <input type="radio" name="prioridade" value="baixa" id="prioridade_baixa">
                        <label for="prioridade_baixa" class="priority-label baixa">
                            <i class="fas fa-circle"></i> Baixa
                        </label>
                    </div>
                    <div class="priority-option">
                        <input type="radio" name="prioridade" value="normal" id="prioridade_normal" checked>
                        <label for="prioridade_normal" class="priority-label normal">
                            <i class="fas fa-circle"></i> Normal
                        </label>
                    </div>
                    <div class="priority-option">
                        <input type="radio" name="prioridade" value="alta" id="prioridade_alta">
                        <label for="prioridade_alta" class="priority-label alta">
                            <i class="fas fa-circle"></i> Alta
                        </label>
                    </div>
                    <div class="priority-option">
                        <input type="radio" name="prioridade" value="urgente" id="prioridade_urgente">
                        <label for="prioridade_urgente" class="priority-label urgente">
                            <i class="fas fa-exclamation-circle"></i> Urgente
                        </label>
                    </div>
                </div>
            </div>

            <button type="submit" name="enviar_mensagem" class="btn-send">
                <i class="fas fa-paper-plane"></i>
                Enviar Mensagem
            </button>
        </form>

        <!-- Lista de Mensagens Enviadas -->
        <div class="form-card">
            <h2 class="form-section-title">
                <i class="fas fa-history"></i>
                Mensagens Enviadas (<?= count($mensagens_enviadas) ?>)
            </h2>

            <div class="messages-list">
                <?php if (empty($mensagens_enviadas)): ?>
                    <div class="empty-state">
                        <i class="fas fa-inbox"></i>
                        <h3>Nenhuma mensagem enviada ainda</h3>
                        <p>Suas mensagens aparecerão aqui</p>
                    </div>
                <?php else: ?>
                    <?php foreach ($mensagens_enviadas as $msg): ?>
                        <div class="message-item">
                            <div class="message-header">
                                <div>
                                    <div class="message-subject">
                                        <?= htmlspecialchars($msg['assunto']) ?>
                                    </div>
                                    <div class="message-meta">
                                        <span>
                                            <i class="fas fa-clock"></i>
                                            <?= date('d/m/Y H:i', strtotime($msg['created_at'])) ?>
                                        </span>
                                        <span>
                                            <i class="fas fa-users"></i>
                                            <?php
                                            $tipos = [
                                                'todos' => 'Todos',
                                                'promoters' => 'Promoters',
                                                'produtores' => 'Produtores',
                                                'evento_ativo' => 'Eventos Ativos',
                                                'evento_especifico' => 'Evento Específico',
                                                'individual' => 'Individual'
                                            ];
                                            echo $tipos[$msg['destinatario_tipo']] ?? 'N/A';
                                            ?>
                                        </span>
                                    </div>
                                </div>
                                <span class="priority-badge <?= $msg['prioridade'] ?>">
                                    <?= strtoupper($msg['prioridade']) ?>
                                </span>
                            </div>

                            <div class="message-body">
                                <?= nl2br(htmlspecialchars($msg['mensagem'])) ?>
                            </div>

                            <div class="message-stats">
                                <div class="stat-item">
                                    <i class="fas fa-paper-plane stat-icon"></i>
                                    <span><?= $msg['total_destinatarios'] ?> enviada(s)</span>
                                </div>
                                <div class="stat-item">
                                    <i class="fas fa-eye stat-icon"></i>
                                    <span><?= $msg['total_lidas'] ?> visualizada(s)</span>
                                </div>
                                <div class="stat-item">
                                    <i class="fas fa-clock stat-icon"></i>
                                    <span><?= $msg['total_destinatarios'] - $msg['total_lidas'] ?> pendente(s)</span>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script>
        // Mostrar/ocultar campos baseado no tipo de destinatário
        document.getElementById('destinatario_tipo').addEventListener('change', function() {
            const valor = this.value;
            const campoEvento = document.getElementById('campo_evento');
            const campoIndividual = document.getElementById('campo_individual');

            campoEvento.style.display = 'none';
            campoIndividual.style.display = 'none';

            if (valor === 'evento_especifico') {
                campoEvento.style.display = 'block';
            } else if (valor === 'individual') {
                campoIndividual.style.display = 'block';
            }
        });
    </script>

    </div><!-- /.main-content -->
    <script src="assets/js/sidebar.js"></script>
</body>
</html>
