<?php
// admin_relatorios.php - SISTEMA DE RELATÓRIOS COM FILTRO POR CATEGORIA DE EVENTO
require_once 'config.php';
verificarRole(['admin']);

$admin_id = $_SESSION['user_id'];

// Criar diretório para relatórios salvos se não existir
$relatorios_dir = 'relatorios_salvos';
if (!file_exists($relatorios_dir)) {
    mkdir($relatorios_dir, 0755, true);
}

// Baixar relatório salvo
if (isset($_GET['download_relatorio'])) {
    $relatorio_id = intval($_GET['download_relatorio']);

    $stmt = $pdo->prepare("SELECT * FROM relatorios_salvos WHERE id = ?");
    $stmt->execute([$relatorio_id]);
    $relatorio = $stmt->fetch();

    if ($relatorio && file_exists($relatorio['arquivo_path'])) {
        header('Content-Type: text/csv; charset=utf-8');
        header('Content-Disposition: attachment; filename="' . $relatorio['arquivo_nome'] . '"');
        readfile($relatorio['arquivo_path']);
        exit;
    } else {
        $_SESSION['erro'] = 'Relatório não encontrado!';
    }
}

// Filtros
$filtro_data_inicio = isset($_GET['data_inicio']) ? $_GET['data_inicio'] : date('Y-m-01');
$filtro_data_fim = isset($_GET['data_fim']) ? $_GET['data_fim'] : date('Y-m-d');
$filtro_evento = isset($_GET['evento_id']) ? intval($_GET['evento_id']) : 0;
$filtro_categoria = isset($_GET['categoria']) ? $_GET['categoria'] : '';

// Buscar estatísticas gerais
$stmt = $pdo->query("SELECT COUNT(*) as total FROM users WHERE role = 'user' AND status = 'aprovado'");
$total_promoters = $stmt->fetch()['total'];

$stmt = $pdo->query("SELECT COUNT(*) as total FROM eventos WHERE status = 'ativo'");
$total_eventos_ativos = $stmt->fetch()['total'];

// Se houver filtro de categoria, ajustar queries
$categoria_condition = '';
$categoria_params = [];
if ($filtro_categoria) {
    if ($filtro_categoria === 'Outros') {
        $categoria_condition = " AND e.event_type = 'Outros'";
    } else {
        $categoria_condition = " AND e.event_type = ?";
        $categoria_params[] = $filtro_categoria;
    }
}

// Total de convidados do período (com filtro de categoria)
if ($filtro_categoria) {
    $stmt = $pdo->prepare("
        SELECT COUNT(c.id) as total 
        FROM convidados c
        INNER JOIN eventos e ON c.evento_id = e.id
        WHERE c.created_at BETWEEN ? AND ?
        $categoria_condition
    ");
    $params = [$filtro_data_inicio . ' 00:00:00', $filtro_data_fim . ' 23:59:59'];
    $params = array_merge($params, $categoria_params);
    $stmt->execute($params);
} else {
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as total 
        FROM convidados 
        WHERE created_at BETWEEN ? AND ?
    ");
    $stmt->execute([$filtro_data_inicio . ' 00:00:00', $filtro_data_fim . ' 23:59:59']);
}
$total_convidados_periodo = $stmt->fetch()['total'];

// Total de check-ins do período (com filtro de categoria)
if ($filtro_categoria) {
    $stmt = $pdo->prepare("
        SELECT COUNT(c.id) as total 
        FROM convidados c
        INNER JOIN eventos e ON c.evento_id = e.id
        WHERE c.checkin = 1 AND c.checkin_data BETWEEN ? AND ?
        $categoria_condition
    ");
    $params = [$filtro_data_inicio . ' 00:00:00', $filtro_data_fim . ' 23:59:59'];
    $params = array_merge($params, $categoria_params);
    $stmt->execute($params);
} else {
    $stmt = $pdo->prepare("
        SELECT COUNT(*) as total 
        FROM convidados 
        WHERE checkin = 1 AND checkin_data BETWEEN ? AND ?
    ");
    $stmt->execute([$filtro_data_inicio . ' 00:00:00', $filtro_data_fim . ' 23:59:59']);
}
$total_checkins_periodo = $stmt->fetch()['total'];

// Comissões do período
$stmt = $pdo->prepare("
    SELECT COALESCE(SUM(valor), 0) as total 
    FROM comissoes 
    WHERE created_at BETWEEN ? AND ?
");
$stmt->execute([$filtro_data_inicio . ' 00:00:00', $filtro_data_fim . ' 23:59:59']);
$total_comissoes_periodo = $stmt->fetch()['total'];

// Saques do período
$stmt = $pdo->prepare("
    SELECT COALESCE(SUM(valor), 0) as total 
    FROM saques 
    WHERE created_at BETWEEN ? AND ? AND status IN ('aprovado', 'concluido')
");
$stmt->execute([$filtro_data_inicio . ' 00:00:00', $filtro_data_fim . ' 23:59:59']);
$total_saques_periodo = $stmt->fetch()['total'];

// Top 10 Promoters do período (com filtro de categoria)
if ($filtro_categoria) {
    $query = "
        SELECT u.id, u.nome, u.instagram,
        (SELECT COUNT(*) FROM convidados c 
         INNER JOIN eventos e ON c.evento_id = e.id 
         WHERE c.promoter_id = u.id AND c.created_at BETWEEN ? AND ? $categoria_condition) as convidados,
        (SELECT COUNT(*) FROM convidados c 
         INNER JOIN eventos e ON c.evento_id = e.id 
         WHERE c.promoter_id = u.id AND c.checkin = 1 AND c.checkin_data BETWEEN ? AND ? $categoria_condition) as checkins,
        (SELECT COALESCE(SUM(co.valor), 0) FROM comissoes co 
         INNER JOIN eventos e ON co.evento_id = e.id
         WHERE co.user_id = u.id AND co.created_at BETWEEN ? AND ? $categoria_condition) as comissoes
        FROM users u
        WHERE u.role = 'user' AND u.status = 'aprovado'
        ORDER BY checkins DESC, convidados DESC
        LIMIT 10
    ";
    $stmt = $pdo->prepare($query);
    $params = [
        $filtro_data_inicio . ' 00:00:00', $filtro_data_fim . ' 23:59:59',
        $filtro_data_inicio . ' 00:00:00', $filtro_data_fim . ' 23:59:59',
        $filtro_data_inicio . ' 00:00:00', $filtro_data_fim . ' 23:59:59'
    ];
    // Adicionar parâmetros de categoria para cada subquery
    if ($filtro_categoria !== 'Outros') {
        $params = [
            $filtro_data_inicio . ' 00:00:00', $filtro_data_fim . ' 23:59:59', $filtro_categoria,
            $filtro_data_inicio . ' 00:00:00', $filtro_data_fim . ' 23:59:59', $filtro_categoria,
            $filtro_data_inicio . ' 00:00:00', $filtro_data_fim . ' 23:59:59', $filtro_categoria
        ];
    }
    $stmt->execute($params);
} else {
    $stmt = $pdo->prepare("
        SELECT u.id, u.nome, u.instagram,
        (SELECT COUNT(*) FROM convidados c WHERE c.promoter_id = u.id AND c.created_at BETWEEN ? AND ?) as convidados,
        (SELECT COUNT(*) FROM convidados c WHERE c.promoter_id = u.id AND c.checkin = 1 AND c.checkin_data BETWEEN ? AND ?) as checkins,
        (SELECT COALESCE(SUM(valor), 0) FROM comissoes co WHERE co.user_id = u.id AND co.created_at BETWEEN ? AND ?) as comissoes
        FROM users u
        WHERE u.role = 'user' AND u.status = 'aprovado'
        ORDER BY checkins DESC, convidados DESC
        LIMIT 10
    ");
    $stmt->execute([
        $filtro_data_inicio . ' 00:00:00', $filtro_data_fim . ' 23:59:59',
        $filtro_data_inicio . ' 00:00:00', $filtro_data_fim . ' 23:59:59',
        $filtro_data_inicio . ' 00:00:00', $filtro_data_fim . ' 23:59:59'
    ]);
}
$top_promoters = $stmt->fetchAll();

// Relatório por evento (com filtro de categoria)
$query = "
    SELECT e.id, e.nome, e.data, e.event_type, e.event_type_other,
    (SELECT COUNT(DISTINCT pe.user_id) FROM produtor_eventos pe WHERE pe.evento_id = e.id) as total_promoters,
    (SELECT COUNT(*) FROM convidados c WHERE c.evento_id = e.id) as total_convidados,
    (SELECT COUNT(*) FROM convidados c WHERE c.evento_id = e.id AND c.checkin = 1) as total_checkins,
    (SELECT COALESCE(SUM(valor), 0) FROM comissoes co WHERE co.evento_id = e.id) as total_comissoes
    FROM eventos e
    WHERE e.data BETWEEN ? AND ?
";

$params = [$filtro_data_inicio, $filtro_data_fim];

if ($filtro_categoria) {
    $query .= $categoria_condition;
    if ($filtro_categoria !== 'Outros') {
        $params[] = $filtro_categoria;
    }
}

$query .= " ORDER BY e.data DESC";

$stmt = $pdo->prepare($query);
$stmt->execute($params);
$eventos_relatorio = $stmt->fetchAll();

// Buscar todos os eventos para o filtro
$stmt = $pdo->query("SELECT id, nome FROM eventos ORDER BY data DESC");
$todos_eventos = $stmt->fetchAll();

// Buscar todas as categorias únicas
$stmt = $pdo->query("
    SELECT DISTINCT event_type 
    FROM eventos 
    WHERE event_type IS NOT NULL AND event_type != ''
    ORDER BY event_type
");
$categorias = $stmt->fetchAll();

// Buscar relatórios salvos (últimos 20)
$relatorios_salvos = [];
try {
    $stmt = $pdo->prepare("
        SELECT r.*, u.nome as admin_nome
        FROM relatorios_salvos r
        LEFT JOIN users u ON r.admin_id = u.id
        ORDER BY r.created_at DESC
        LIMIT 20
    ");
    $stmt->execute();
    $relatorios_salvos = $stmt->fetchAll();
} catch (PDOException $e) {
    // Tabela ainda não existe
    error_log("Tabela relatorios_salvos não existe: " . $e->getMessage());
}

// Gráfico de check-ins por dia (últimos 30 dias)
$stmt = $pdo->query("
    SELECT DATE(checkin_data) as data, COUNT(*) as total
    FROM convidados
    WHERE checkin = 1 AND checkin_data >= DATE_SUB(NOW(), INTERVAL 30 DAY)
    GROUP BY DATE(checkin_data)
    ORDER BY data ASC
");
$checkins_por_dia = $stmt->fetchAll();

// Exportar lista de convidados FILTRADO (GERAL - COM TODOS OS FILTROS)
if (isset($_GET['exportar_filtrado'])) {
    // Nome do arquivo baseado nos filtros
    $arquivo_nome_parts = ['relatorio_convidados'];
    if ($filtro_categoria) {
        $arquivo_nome_parts[] = str_replace(' ', '_', $filtro_categoria);
    }
    $arquivo_nome_parts[] = date('Y-m-d_His');
    $arquivo_nome = implode('_', $arquivo_nome_parts) . '.csv';
    $arquivo_path = $relatorios_dir . '/' . $arquivo_nome;

    // Criar arquivo
    $output = fopen($arquivo_path, 'w');
    fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF)); // BOM para UTF-8

    // Cabeçalho
    fputcsv($output, ['Nome', 'Telefone', 'WhatsApp', 'Evento', 'Categoria', 'Data Evento', 'Promoter', 'Check-in', 'Data Check-in', 'Criado em'], ';');

    // Construir query baseada nos filtros
    $query = "
        SELECT
            c.nome as convidado_nome,
            c.whatsapp as telefone,
            c.whatsapp as whatsapp_link,
            e.nome as evento_nome,
            e.event_type as categoria,
            e.data as evento_data,
            u.nome as promoter_nome,
            c.checkin,
            c.checkin_data,
            c.created_at
        FROM convidados c
        INNER JOIN eventos e ON c.evento_id = e.id
        LEFT JOIN users u ON c.promoter_id = u.id
        WHERE 1=1
    ";

    $params = [];

    // Filtro de data
    if ($filtro_data_inicio && $filtro_data_fim) {
        $query .= " AND c.created_at BETWEEN ? AND ?";
        $params[] = $filtro_data_inicio . ' 00:00:00';
        $params[] = $filtro_data_fim . ' 23:59:59';
    }

    // Filtro de categoria
    if ($filtro_categoria) {
        if ($filtro_categoria === 'Outros') {
            $query .= " AND e.event_type = 'Outros'";
        } else {
            $query .= " AND e.event_type = ?";
            $params[] = $filtro_categoria;
        }
    }

    $query .= " ORDER BY c.created_at DESC";

    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
    $rows = $stmt->fetchAll();

    $total_registros = 0;
    foreach ($rows as $row) {
        // Formatar WhatsApp para link
        $telefone_limpo = preg_replace('/[^0-9]/', '', $row['whatsapp_link']);
        if (!str_starts_with($telefone_limpo, '55')) {
            $telefone_limpo = '55' . $telefone_limpo;
        }

        fputcsv($output, [
            $row['convidado_nome'],
            $row['telefone'],
            'https://wa.me/' . $telefone_limpo,
            $row['evento_nome'],
            $row['categoria'] ?: '-',
            date('d/m/Y', strtotime($row['evento_data'])),
            $row['promoter_nome'] ?: 'N/A',
            $row['checkin'] ? 'Sim' : 'Não',
            $row['checkin_data'] ? date('d/m/Y H:i', strtotime($row['checkin_data'])) : '-',
            date('d/m/Y H:i', strtotime($row['created_at']))
        ], ';');
        $total_registros++;
    }

    fclose($output);

    // Salvar registro no banco de dados
    try {
        $stmt = $pdo->prepare("
            INSERT INTO relatorios_salvos
            (admin_id, tipo, categoria, data_inicio, data_fim, arquivo_nome, arquivo_path, total_registros)
            VALUES (?, 'relatorio_filtrado', ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $admin_id,
            $filtro_categoria ?: 'Todas',
            $filtro_data_inicio,
            $filtro_data_fim,
            $arquivo_nome,
            $arquivo_path,
            $total_registros
        ]);
    } catch (PDOException $e) {
        // Tabela ainda não existe, continuar sem salvar registro
        error_log("Erro ao salvar registro de relatório: " . $e->getMessage());
    }

    // Enviar arquivo para download
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $arquivo_nome . '"');
    readfile($arquivo_path);
    exit;
}

// Exportar lista de convidados por categoria (COM SALVAMENTO) - LEGADO
if (isset($_GET['exportar_categoria']) && $filtro_categoria) {
    // Nome do arquivo
    $arquivo_nome = 'convidados_' . str_replace(' ', '_', $filtro_categoria) . '_' . date('Y-m-d_His') . '.csv';
    $arquivo_path = $relatorios_dir . '/' . $arquivo_nome;

    // Criar arquivo
    $output = fopen($arquivo_path, 'w');
    fprintf($output, chr(0xEF).chr(0xBB).chr(0xBF)); // BOM para UTF-8

    // Cabeçalho
    fputcsv($output, ['Nome', 'Telefone', 'Evento', 'Data Evento', 'Promoter', 'Check-in', 'Data Check-in'], ';');

    // Buscar convidados da categoria
    $query = "
        SELECT
            c.nome as convidado_nome,
            c.whatsapp as telefone,
            e.nome as evento_nome,
            e.data as evento_data,
            u.nome as promoter_nome,
            c.checkin,
            c.checkin_data
        FROM convidados c
        INNER JOIN eventos e ON c.evento_id = e.id
        LEFT JOIN users u ON c.promoter_id = u.id
        WHERE e.event_type = ?
    ";

    $params = [$filtro_categoria];

    if ($filtro_data_inicio && $filtro_data_fim) {
        $query .= " AND c.created_at BETWEEN ? AND ?";
        $params[] = $filtro_data_inicio . ' 00:00:00';
        $params[] = $filtro_data_fim . ' 23:59:59';
    }

    $query .= " ORDER BY c.created_at DESC";

    $stmt = $pdo->prepare($query);
    $stmt->execute($params);
    $rows = $stmt->fetchAll();

    $total_registros = 0;
    foreach ($rows as $row) {
        fputcsv($output, [
            $row['convidado_nome'],
            $row['telefone'],
            $row['evento_nome'],
            date('d/m/Y', strtotime($row['evento_data'])),
            $row['promoter_nome'] ?: 'N/A',
            $row['checkin'] ? 'Sim' : 'Não',
            $row['checkin_data'] ? date('d/m/Y H:i', strtotime($row['checkin_data'])) : '-'
        ], ';');
        $total_registros++;
    }

    fclose($output);

    // Salvar registro no banco de dados
    try {
        $stmt = $pdo->prepare("
            INSERT INTO relatorios_salvos
            (admin_id, tipo, categoria, data_inicio, data_fim, arquivo_nome, arquivo_path, total_registros)
            VALUES (?, 'convidados_categoria', ?, ?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $admin_id,
            $filtro_categoria,
            $filtro_data_inicio,
            $filtro_data_fim,
            $arquivo_nome,
            $arquivo_path,
            $total_registros
        ]);
    } catch (PDOException $e) {
        // Tabela ainda não existe, continuar sem salvar registro
        error_log("Erro ao salvar registro de relatório: " . $e->getMessage());
    }

    // Enviar arquivo para download
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="' . $arquivo_nome . '"');
    readfile($arquivo_path);
    exit;
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Relatórios - Admin</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="assets/dashboard-style.css">
    <link rel="stylesheet" href="assets/css/sidebar.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        
        :root[data-theme="dark"] {
            --bg-primary: #000000;
            --bg-secondary: #1a1a1a;
            --bg-card: #2a2a2a;
            --color-primary: #FFD700;
            --color-text: #ffffff;
            --color-text-secondary: #999999;
            --border-color: #333333;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: var(--bg-secondary);
            color: var(--color-text);
        }
.container {
            max-width: 1400px;
            margin: 30px auto;
            padding: 0 20px;
        }
        
        .page-title {
            font-size: 32px;
            color: var(--color-primary);
            margin-bottom: 30px;
        }
        
        .filters-section {
            background: var(--bg-card);
            padding: 20px;
            border-radius: 12px;
            margin-bottom: 30px;
            border: 1px solid var(--border-color);
        }
        
        .filters-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 20px;
        }
        
        .filter-group {
            display: flex;
            flex-direction: column;
        }
        
        .filter-group label {
            color: var(--color-text-secondary);
            margin-bottom: 5px;
            font-size: 14px;
        }
        
        .filter-group input,
        .filter-group select {
            padding: 8px;
            border: 1px solid var(--border-color);
            border-radius: 8px;
            background: var(--bg-secondary);
            color: var(--color-text);
        }
        
        .filter-actions {
            display: flex;
            gap: 10px;
            flex-wrap: wrap;
        }
        
        .btn {
            padding: 10px 20px;
            border: none;
            border-radius: 8px;
            cursor: pointer;
            text-decoration: none;
            font-size: 14px;
            transition: all 0.3s;
            display: inline-block;
        }
        
        .btn-primary {
            background: var(--color-primary);
            color: #000;
        }
        
        .btn-success {
            background: #4CAF50;
            color: white;
        }
        
        .btn-secondary {
            background: var(--bg-card);
            color: var(--color-text);
            border: 1px solid var(--border-color);
        }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(255, 215, 0, 0.2);
        }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 40px;
        }
        
        .stat-card {
            background: var(--bg-card);
            padding: 25px;
            border-radius: 12px;
            border: 1px solid var(--border-color);
            transition: all 0.3s;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.5);
        }
        
        .stat-icon {
            font-size: 36px;
            margin-bottom: 10px;
        }
        
        .stat-value {
            font-size: 32px;
            font-weight: bold;
            color: var(--color-primary);
            margin-bottom: 5px;
        }
        
        .stat-label {
            color: var(--color-text-secondary);
            font-size: 14px;
        }
        
        .section-card {
            background: var(--bg-card);
            padding: 25px;
            border-radius: 12px;
            margin-bottom: 30px;
            border: 1px solid var(--border-color);
        }
        
        .section-title {
            font-size: 24px;
            color: var(--color-primary);
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .table {
            width: 100%;
            border-collapse: collapse;
        }
        
        .table th {
            background: var(--bg-secondary);
            padding: 12px;
            text-align: left;
            color: var(--color-primary);
            border-bottom: 2px solid var(--border-color);
        }
        
        .table td {
            padding: 12px;
            border-bottom: 1px solid var(--border-color);
        }
        
        .table tr:hover {
            background: rgba(255, 215, 0, 0.05);
        }
        
        .badge {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: bold;
        }
        
        .badge-success {
            background: rgba(76, 175, 80, 0.2);
            color: #4CAF50;
        }
        
        .badge-warning {
            background: rgba(255, 152, 0, 0.2);
            color: #FF9800;
        }
        
        .badge-danger {
            background: rgba(244, 67, 54, 0.2);
            color: #F44336;
        }
        
        .chart-container {
            position: relative;
            height: 400px;
            margin-top: 20px;
        }
        
        /* NOVO: Estilos para filtro de categoria */
        .category-badge {
            display: inline-flex;
            align-items: center;
            padding: 4px 10px;
            background: rgba(255, 215, 0, 0.2);
            color: var(--color-primary);
            border-radius: 15px;
            font-size: 12px;
            margin-left: 10px;
        }
        
        .category-icon {
            margin-right: 6px;
            font-size: 14px;
        }
        
        .filter-category-select {
            min-width: 200px;
        }
        
        .export-category-btn {
            background: #4CAF50;
            color: white;
            display: inline-flex;
            align-items: center;
            gap: 8px;
        }
        
        @media (max-width: 768px) {
            body {
                padding: 10px;
            }

            .container {
                max-width: 100%;
                margin: 10px auto;
                padding: 0 10px;
            }
.stats-grid {
                grid-template-columns: 1fr;
                gap: 15px;
            }

            .filters-grid {
                grid-template-columns: 1fr;
            }

            .filters-section {
                padding: 15px;
                margin-bottom: 20px;
            }

            .section-card {
                padding: 15px;
                margin-bottom: 20px;
                overflow-x: auto;
            }

            .table-container {
                overflow-x: auto;
                margin: 0 -15px;
                padding: 0 15px;
            }

            .table {
                font-size: 12px;
                min-width: 100%;
            }

            .table th, .table td {
                padding: 6px;
                font-size: 12px;
            }

            .page-title {
                font-size: 24px;
            }

            .section-title {
                font-size: 18px;
                flex-wrap: wrap;
            }

            .filter-actions {
                flex-direction: column;
            }

            .btn {
                width: 100%;
                text-align: center;
            }

            .chart-container {
                height: 300px;
            }
        }
    </style>
</head>
<body>
    <?php include 'assets/sidebar.php'; ?>

    <div class="container">
        <h1 class="page-title"><i class="fas fa-chart-bar"></i> Relatórios e Estatísticas</h1>
        
        <!-- Seção de Filtros -->
        <div class="filters-section">
            <form method="GET" action="">
                <div class="filters-grid">
                    <div class="filter-group">
                        <label>Data Início</label>
                        <input type="date" name="data_inicio" value="<?php echo $filtro_data_inicio; ?>">
                    </div>
                    <div class="filter-group">
                        <label>Data Fim</label>
                        <input type="date" name="data_fim" value="<?php echo $filtro_data_fim; ?>">
                    </div>
                    <div class="filter-group">
                        <label>Evento Específico</label>
                        <select name="evento_id">
                            <option value="0">Todos os Eventos</option>
                            <?php foreach ($todos_eventos as $evento): ?>
                                <option value="<?php echo $evento['id']; ?>" <?php echo $filtro_evento == $evento['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($evento['nome']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <!-- NOVO: Filtro por categoria -->
                    <div class="filter-group">
                        <label>Categoria de Evento</label>
                        <select name="categoria" class="filter-category-select">
                            <option value="">Todas as Categorias</option>
                            <option value="Pagode" <?php echo $filtro_categoria == 'Pagode' ? 'selected' : ''; ?>>Pagode</option>
                            <option value="Sertanejo" <?php echo $filtro_categoria == 'Sertanejo' ? 'selected' : ''; ?>>Sertanejo</option>
                            <option value="Eletronico" <?php echo $filtro_categoria == 'Eletronico' ? 'selected' : ''; ?>>Eletrônico</option>
                            <option value="Rock" <?php echo $filtro_categoria == 'Rock' ? 'selected' : ''; ?>>Rock</option>
                            <option value="Funk" <?php echo $filtro_categoria == 'Funk' ? 'selected' : ''; ?>>Funk</option>
                            <option value="Matine" <?php echo $filtro_categoria == 'Matine' ? 'selected' : ''; ?>>Matinê</option>
                            <option value="Gospel" <?php echo $filtro_categoria == 'Gospel' ? 'selected' : ''; ?>>Gospel</option>
                            <option value="Trap" <?php echo $filtro_categoria == 'Trap' ? 'selected' : ''; ?>>Trap</option>
                            <option value="Rap" <?php echo $filtro_categoria == 'Rap' ? 'selected' : ''; ?>>Rap</option>
                            <option value="Outros" <?php echo $filtro_categoria == 'Outros' ? 'selected' : ''; ?>>Outros</option>
                        </select>
                    </div>
                </div>
                <div class="filter-actions">
                    <button type="submit" class="btn btn-primary"><i class="fas fa-filter"></i> Filtrar</button>
                    <a href="admin_relatorios.php" class="btn btn-secondary"><i class="fas fa-redo"></i> Limpar Filtros</a>

                    <!-- Botão de exportação geral (sempre visível) -->
                    <a href="?exportar_filtrado=1&categoria=<?php echo urlencode($filtro_categoria); ?>&data_inicio=<?php echo $filtro_data_inicio; ?>&data_fim=<?php echo $filtro_data_fim; ?>"
                       class="btn btn-success export-category-btn"
                       title="Exportar relatório com base nos filtros aplicados">
                        <i class="fas fa-file-excel"></i> Exportar Excel/CSV
                        <?php if ($filtro_categoria): ?>
                            (<?php echo htmlspecialchars($filtro_categoria); ?>)
                        <?php endif; ?>
                    </a>
                </div>
            </form>
        </div>
        
        <?php if ($filtro_categoria): ?>
            <div style="background: rgba(255, 215, 0, 0.1); padding: 15px; border-radius: 8px; margin-bottom: 20px; border: 1px solid var(--color-primary);">
                <strong>Filtrando por categoria:</strong>
                <span class="category-badge">
                    <span class="category-icon"><i class="fas fa-music"></i></span>
                    <?php echo htmlspecialchars($filtro_categoria); ?>
                </span>
            </div>
        <?php endif; ?>
        
        <!-- Cards de Estatísticas -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-icon"><i class="fas fa-users"></i></div>
                <div class="stat-value"><?php echo number_format($total_promoters); ?></div>
                <div class="stat-label">Promoters Aprovados</div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon"><i class="fas fa-glass-cheers"></i></div>
                <div class="stat-value"><?php echo number_format($total_eventos_ativos); ?></div>
                <div class="stat-label">Eventos Ativos</div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon"><i class="fas fa-user-friends"></i></div>
                <div class="stat-value"><?php echo number_format($total_convidados_periodo); ?></div>
                <div class="stat-label">Convidados no Período<?php echo $filtro_categoria ? ' (' . $filtro_categoria . ')' : ''; ?></div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon"><i class="fas fa-check-circle"></i></div>
                <div class="stat-value"><?php echo number_format($total_checkins_periodo); ?></div>
                <div class="stat-label">Check-ins no Período<?php echo $filtro_categoria ? ' (' . $filtro_categoria . ')' : ''; ?></div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon"><i class="fas fa-money-bill-wave"></i></div>
                <div class="stat-value">R$ <?php echo number_format($total_comissoes_periodo, 2, ',', '.'); ?></div>
                <div class="stat-label">Comissões no Período</div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon"><i class="fas fa-hand-holding-usd"></i></div>
                <div class="stat-value">R$ <?php echo number_format($total_saques_periodo, 2, ',', '.'); ?></div>
                <div class="stat-label">Saques no Período</div>
            </div>
        </div>
        
        <!-- Top Promoters -->
        <div class="section-card">
            <h2 class="section-title">
                <i class="fas fa-trophy"></i> Top 10 Promoters
                <?php if ($filtro_categoria): ?>
                    <span class="category-badge">
                        <span class="category-icon"><i class="fas fa-music"></i></span>
                        <?php echo htmlspecialchars($filtro_categoria); ?>
                    </span>
                <?php endif; ?>
            </h2>
            <table class="table">
                <thead>
                    <tr>
                        <th>Pos.</th>
                        <th>Nome</th>
                        <th>Instagram</th>
                        <th>Convidados</th>
                        <th>Check-ins</th>
                        <th>Taxa de Conversão</th>
                        <th>Comissões</th>
                    </tr>
                </thead>
                <tbody>
                    <?php 
                    $posicao = 1;
                    foreach ($top_promoters as $promoter): 
                        $taxa_conversao = $promoter['convidados'] > 0 
                            ? round(($promoter['checkins'] / $promoter['convidados']) * 100, 1)
                            : 0;
                    ?>
                    <tr>
                        <td><strong><?php echo $posicao++; ?>º</strong></td>
                        <td><?php echo htmlspecialchars($promoter['nome']); ?></td>
                        <td><?php echo htmlspecialchars($promoter['instagram']); ?></td>
                        <td><?php echo number_format($promoter['convidados']); ?></td>
                        <td><?php echo number_format($promoter['checkins']); ?></td>
                        <td>
                            <span class="badge <?php echo $taxa_conversao > 50 ? 'badge-success' : ($taxa_conversao > 30 ? 'badge-warning' : 'badge-danger'); ?>">
                                <?php echo $taxa_conversao; ?>%
                            </span>
                        </td>
                        <td>R$ <?php echo number_format($promoter['comissoes'], 2, ',', '.'); ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        
        <!-- Relatório por Evento -->
        <div class="section-card">
            <h2 class="section-title">
                <i class="fas fa-calendar-alt"></i> Relatório por Evento
                <?php if ($filtro_categoria): ?>
                    <span class="category-badge">
                        <span class="category-icon"><i class="fas fa-music"></i></span>
                        <?php echo htmlspecialchars($filtro_categoria); ?>
                    </span>
                <?php endif; ?>
            </h2>
            <table class="table">
                <thead>
                    <tr>
                        <th>Evento</th>
                        <th>Categoria</th>
                        <th>Data</th>
                        <th>Promoters</th>
                        <th>Convidados</th>
                        <th>Check-ins</th>
                        <th>Taxa de Conversão</th>
                        <th>Comissões</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($eventos_relatorio as $evento): 
                        $taxa_conversao = $evento['total_convidados'] > 0 
                            ? round(($evento['total_checkins'] / $evento['total_convidados']) * 100, 1)
                            : 0;
                        
                        $categoria_display = $evento['event_type'];
                        if ($evento['event_type'] === 'Outros' && $evento['event_type_other']) {
                            $categoria_display = $evento['event_type_other'];
                        }
                    ?>
                    <tr>
                        <td><?php echo htmlspecialchars($evento['nome']); ?></td>
                        <td>
                            <?php if ($categoria_display): ?>
                                <span style="color: var(--color-primary);"><i class="fas fa-music"></i> <?php echo htmlspecialchars($categoria_display); ?></span>
                            <?php else: ?>
                                <span style="color: var(--color-text-secondary);">-</span>
                            <?php endif; ?>
                        </td>
                        <td><?php echo date('d/m/Y', strtotime($evento['data'])); ?></td>
                        <td><?php echo number_format($evento['total_promoters']); ?></td>
                        <td><?php echo number_format($evento['total_convidados']); ?></td>
                        <td><?php echo number_format($evento['total_checkins']); ?></td>
                        <td>
                            <span class="badge <?php echo $taxa_conversao > 50 ? 'badge-success' : ($taxa_conversao > 30 ? 'badge-warning' : 'badge-danger'); ?>">
                                <?php echo $taxa_conversao; ?>%
                            </span>
                        </td>
                        <td>R$ <?php echo number_format($evento['total_comissoes'], 2, ',', '.'); ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>

        <!-- Relatórios Salvos -->
        <?php if (count($relatorios_salvos) > 0): ?>
        <div class="section-card">
            <h2 class="section-title">
                <i class="fas fa-history"></i> Relatórios Salvos (Download)
            </h2>
            <table class="table">
                <thead>
                    <tr>
                        <th>Data</th>
                        <th>Tipo</th>
                        <th>Categoria</th>
                        <th>Período</th>
                        <th>Registros</th>
                        <th>Gerado Por</th>
                        <th>Ações</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($relatorios_salvos as $rel): ?>
                    <tr>
                        <td><?php echo date('d/m/Y H:i', strtotime($rel['created_at'])); ?></td>
                        <td>
                            <span class="badge" style="background: #17a2b8;">
                                <?php
                                $tipos = [
                                    'convidados_categoria' => 'Convidados por Categoria',
                                    'relatorio_filtrado' => 'Relatório Filtrado',
                                    'evento_completo' => 'Evento Completo',
                                    'geral' => 'Relatório Geral'
                                ];
                                echo $tipos[$rel['tipo']] ?? $rel['tipo'];
                                ?>
                            </span>
                        </td>
                        <td><?php echo htmlspecialchars($rel['categoria'] ?? '-'); ?></td>
                        <td>
                            <?php
                            if ($rel['data_inicio'] && $rel['data_fim']) {
                                echo date('d/m/Y', strtotime($rel['data_inicio'])) . ' até ' . date('d/m/Y', strtotime($rel['data_fim']));
                            } else {
                                echo '-';
                            }
                            ?>
                        </td>
                        <td><?php echo number_format($rel['total_registros'], 0, ',', '.'); ?></td>
                        <td><?php echo htmlspecialchars($rel['admin_nome'] ?? 'N/A'); ?></td>
                        <td>
                            <a href="?download_relatorio=<?php echo $rel['id']; ?>" class="btn-action" title="Baixar">
                                <i class="fas fa-download"></i> Baixar
                            </a>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        </div>
        <?php endif; ?>

        <!-- Gráfico de Check-ins por Dia -->
        <div class="section-card">
            <h2 class="section-title"><i class="fas fa-chart-line"></i> Check-ins dos Últimos 30 Dias</h2>
            <div class="chart-container">
                <canvas id="checkinsChart"></canvas>
            </div>
        </div>
    </div>
    
    <script>
        const savedTheme = localStorage.getItem('theme') || 'dark';
        document.documentElement.setAttribute('data-theme', savedTheme);
        
        // Gráfico de check-ins
        const ctx = document.getElementById('checkinsChart').getContext('2d');
        const checkinsData = <?php echo json_encode($checkins_por_dia); ?>;
        
        new Chart(ctx, {
            type: 'line',
            data: {
                labels: checkinsData.map(d => {
                    const date = new Date(d.data);
                    return date.toLocaleDateString('pt-BR', { day: '2-digit', month: '2-digit' });
                }),
                datasets: [{
                    label: 'Check-ins',
                    data: checkinsData.map(d => d.total),
                    borderColor: '#FFD700',
                    backgroundColor: 'rgba(255, 215, 0, 0.1)',
                    tension: 0.3,
                    fill: true
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        display: false
                    }
                },
                scales: {
                    y: {
                        beginAtZero: true,
                        grid: {
                            color: 'rgba(255, 255, 255, 0.1)'
                        },
                        ticks: {
                            color: '#999'
                        }
                    },
                    x: {
                        grid: {
                            color: 'rgba(255, 255, 255, 0.1)'
                        },
                        ticks: {
                            color: '#999'
                        }
                    }
                }
            }
        });
    </script>
    </div><!-- /.main-content -->
    <script src="assets/js/sidebar.js"></script>
</body>
</html>