<?php
// api/notificacoes_api.php - API de Notificações v3.0
header('Content-Type: application/json');
require_once '../config.php';

// Verificar se está logado
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Não autorizado']);
    exit;
}

$user_id = $_SESSION['user_id'];
$action = $_GET['action'] ?? '';

try {
    switch ($action) {
        case 'contar':
            // Contar notificações não lidas
            $stmt = $pdo->prepare("
                SELECT
                    COUNT(*) as total,
                    COALESCE(SUM(CASE WHEN lida = 0 THEN 1 ELSE 0 END), 0) as nao_lidas,
                    COALESCE(SUM(CASE WHEN categoria = 'financeiro' AND lida = 0 THEN 1 ELSE 0 END), 0) as financeiro,
                    COALESCE(SUM(CASE WHEN categoria = 'eventos' AND lida = 0 THEN 1 ELSE 0 END), 0) as eventos,
                    COALESCE(SUM(CASE WHEN categoria = 'geral' AND lida = 0 THEN 1 ELSE 0 END), 0) as geral,
                    COALESCE(SUM(CASE WHEN categoria = 'mensagens' AND lida = 0 THEN 1 ELSE 0 END), 0) as mensagens
                FROM notificacoes
                WHERE user_id = ?
            ");
            $stmt->execute([$user_id]);
            $contadores = $stmt->fetch(PDO::FETCH_ASSOC);

            echo json_encode([
                'success' => true,
                'nao_lidas' => (int)$contadores['nao_lidas'],
                'contadores' => $contadores
            ]);
            break;

        case 'listar':
            // Listar notificações
            $categoria = $_GET['categoria'] ?? null;
            $limit = min((int)($_GET['limit'] ?? 20), 100);
            $offset = max((int)($_GET['offset'] ?? 0), 0);

            $where = "user_id = ?";
            $params = [$user_id];

            if ($categoria && $categoria !== 'todas') {
                $where .= " AND categoria = ?";
                $params[] = $categoria;
            }

            $stmt = $pdo->prepare("
                SELECT *
                FROM notificacoes
                WHERE $where
                ORDER BY lida ASC, created_at DESC
                LIMIT $limit OFFSET $offset
            ");
            $stmt->execute($params);
            $notificacoes = $stmt->fetchAll(PDO::FETCH_ASSOC);

            echo json_encode([
                'success' => true,
                'notificacoes' => $notificacoes,
                'total' => count($notificacoes)
            ]);
            break;

        case 'marcar_lida':
            // Marcar notificação como lida
            $notif_id = (int)($_POST['id'] ?? 0);

            if (!$notif_id) {
                throw new Exception('ID da notificação não fornecido');
            }

            $stmt = $pdo->prepare("
                UPDATE notificacoes
                SET lida = 1, lida_em = NOW()
                WHERE id = ? AND user_id = ?
            ");
            $stmt->execute([$notif_id, $user_id]);

            // Atualizar contador
            $stmt = $pdo->prepare("
                UPDATE users
                SET notificacoes_nao_lidas = GREATEST(notificacoes_nao_lidas - 1, 0)
                WHERE id = ?
            ");
            $stmt->execute([$user_id]);

            echo json_encode([
                'success' => true,
                'message' => 'Notificação marcada como lida'
            ]);
            break;

        case 'marcar_todas_lidas':
            // Marcar todas como lidas
            $categoria = $_POST['categoria'] ?? null;

            if ($categoria && $categoria !== 'todas') {
                $stmt = $pdo->prepare("
                    UPDATE notificacoes
                    SET lida = 1, lida_em = NOW()
                    WHERE user_id = ? AND categoria = ? AND lida = 0
                ");
                $stmt->execute([$user_id, $categoria]);
            } else {
                $stmt = $pdo->prepare("
                    UPDATE notificacoes
                    SET lida = 1, lida_em = NOW()
                    WHERE user_id = ? AND lida = 0
                ");
                $stmt->execute([$user_id]);
            }

            // Recalcular contador
            $stmt = $pdo->prepare("SELECT COUNT(*) FROM notificacoes WHERE user_id = ? AND lida = 0");
            $stmt->execute([$user_id]);
            $nao_lidas = $stmt->fetchColumn();

            $pdo->prepare("UPDATE users SET notificacoes_nao_lidas = ? WHERE id = ?")
                ->execute([$nao_lidas, $user_id]);

            echo json_encode([
                'success' => true,
                'message' => 'Todas as notificações foram marcadas como lidas',
                'nao_lidas' => $nao_lidas
            ]);
            break;

        case 'ultimas':
            // Buscar últimas notificações não lidas
            $limit = min((int)($_GET['limit'] ?? 5), 20);

            $stmt = $pdo->prepare("
                SELECT *
                FROM notificacoes
                WHERE user_id = ? AND lida = 0
                ORDER BY created_at DESC
                LIMIT $limit
            ");
            $stmt->execute([$user_id]);
            $notificacoes = $stmt->fetchAll(PDO::FETCH_ASSOC);

            echo json_encode([
                'success' => true,
                'notificacoes' => $notificacoes
            ]);
            break;

        default:
            throw new Exception('Ação não reconhecida');
    }

} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
