<?php
// api/status_api.php - API de Status Online v3.0
header('Content-Type: application/json');
require_once '../config.php';

// Verificar se está logado
if (!isset($_SESSION['user_id'])) {
    http_response_code(401);
    echo json_encode(['error' => 'Não autorizado']);
    exit;
}

$user_id = $_SESSION['user_id'];
$action = $_GET['action'] ?? $_POST['action'] ?? '';

try {
    switch ($action) {
        case 'heartbeat':
            // Atualizar status como online
            $stmt = $pdo->prepare("
                INSERT INTO user_status (user_id, status, ultima_atividade)
                VALUES (?, 'online', NOW())
                ON DUPLICATE KEY UPDATE
                    status = 'online',
                    ultima_atividade = NOW()
            ");
            $stmt->execute([$user_id]);

            echo json_encode([
                'success' => true,
                'status' => 'online'
            ]);
            break;

        case 'set_status':
            // Definir status manualmente (online, ausente, offline)
            $status = $_POST['status'] ?? 'online';

            if (!in_array($status, ['online', 'ausente', 'offline'])) {
                throw new Exception('Status inválido');
            }

            $stmt = $pdo->prepare("
                INSERT INTO user_status (user_id, status, ultima_atividade)
                VALUES (?, ?, NOW())
                ON DUPLICATE KEY UPDATE
                    status = ?,
                    ultima_atividade = NOW()
            ");
            $stmt->execute([$user_id, $status, $status]);

            echo json_encode([
                'success' => true,
                'status' => $status
            ]);
            break;

        case 'get_status':
            // Obter status de um usuário
            $target_user_id = (int)($_GET['user_id'] ?? 0);

            if (!$target_user_id) {
                throw new Exception('ID do usuário não fornecido');
            }

            $stmt = $pdo->prepare("
                SELECT
                    us.*,
                    u.nome,
                    u.role,
                    CASE
                        WHEN us.ultima_atividade >= DATE_SUB(NOW(), INTERVAL 2 MINUTE) THEN 'online'
                        WHEN us.ultima_atividade >= DATE_SUB(NOW(), INTERVAL 5 MINUTE) THEN 'ausente'
                        ELSE 'offline'
                    END as status_calculado
                FROM user_status us
                INNER JOIN users u ON us.user_id = u.id
                WHERE us.user_id = ?
            ");
            $stmt->execute([$target_user_id]);
            $status = $stmt->fetch(PDO::FETCH_ASSOC);

            echo json_encode([
                'success' => true,
                'status' => $status
            ]);
            break;

        case 'online_users':
            // Listar usuários online
            $role_filter = $_GET['role'] ?? null;

            $where = "us.ultima_atividade >= DATE_SUB(NOW(), INTERVAL 5 MINUTE)";
            $params = [];

            if ($role_filter) {
                $where .= " AND u.role = ?";
                $params[] = $role_filter;
            }

            $stmt = $pdo->prepare("
                SELECT
                    us.user_id,
                    u.nome,
                    u.role,
                    u.foto_perfil,
                    CASE
                        WHEN us.ultima_atividade >= DATE_SUB(NOW(), INTERVAL 2 MINUTE) THEN 'online'
                        WHEN us.ultima_atividade >= DATE_SUB(NOW(), INTERVAL 5 MINUTE) THEN 'ausente'
                        ELSE 'offline'
                    END as status,
                    us.ultima_atividade
                FROM user_status us
                INNER JOIN users u ON us.user_id = u.id
                WHERE $where
                ORDER BY us.ultima_atividade DESC
            ");
            $stmt->execute($params);
            $usuarios = $stmt->fetchAll(PDO::FETCH_ASSOC);

            echo json_encode([
                'success' => true,
                'usuarios' => $usuarios,
                'total' => count($usuarios)
            ]);
            break;

        case 'count_online':
            // Contar usuários online
            $stmt = $pdo->query("
                SELECT
                    COUNT(*) as total,
                    SUM(CASE WHEN us.ultima_atividade >= DATE_SUB(NOW(), INTERVAL 2 MINUTE) THEN 1 ELSE 0 END) as online,
                    SUM(CASE WHEN us.ultima_atividade >= DATE_SUB(NOW(), INTERVAL 5 MINUTE)
                        AND us.ultima_atividade < DATE_SUB(NOW(), INTERVAL 2 MINUTE) THEN 1 ELSE 0 END) as ausente
                FROM user_status us
                WHERE us.ultima_atividade >= DATE_SUB(NOW(), INTERVAL 5 MINUTE)
            ");
            $contadores = $stmt->fetch(PDO::FETCH_ASSOC);

            echo json_encode([
                'success' => true,
                'contadores' => $contadores
            ]);
            break;

        default:
            throw new Exception('Ação não reconhecida');
    }

} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
}
