// assets/js/notificacoes.js - Sistema de Notificações em Tempo Real v3.0

(function() {
    'use strict';

    // Configurações
    const CONFIG = {
        POLLING_INTERVAL: 10000,      // 10 segundos
        HEARTBEAT_INTERVAL: 30000,    // 30 segundos
        TOAST_DURATION: 5000,         // 5 segundos
        SOUND_ENABLED: true
    };

    // Estado
    let lastMessageCount = 0;
    let lastNotificationCount = 0;
    let pollingTimer = null;
    let heartbeatTimer = null;

    /**
     * Inicializar sistema de notificações
     */
    function init() {
        console.log('[Notificações] Sistema iniciado');

        // Buscar contadores iniciais
        updateCounters();

        // Iniciar polling de contadores
        startPolling();

        // Iniciar heartbeat de status
        startHeartbeat();

        // Event listeners
        setupEventListeners();

        // Limpar timers ao sair da página
        window.addEventListener('beforeunload', cleanup);
    }

    /**
     * Atualizar badges de contadores
     */
    async function updateCounters() {
        try {
            // Buscar contador de mensagens
            const msgResponse = await fetch('api/mensagens_api.php?action=contar');
            const msgData = await msgResponse.json();

            console.log('[Notificações] Resposta de mensagens:', msgData);

            if (msgData.success) {
                // Suportar ambos formatos: msgData.nao_lidas ou msgData.contadores.nao_lidas
                const naoLidas = parseInt(msgData.nao_lidas || msgData.contadores?.nao_lidas) || 0;

                console.log('[Notificações] Mensagens não lidas:', naoLidas);
                updateBadge('mensagens', naoLidas);

                // Verificar se há novas mensagens
                if (naoLidas > lastMessageCount && lastMessageCount > 0) {
                    const newMessages = naoLidas - lastMessageCount;
                    showToast(
                        'Nova Mensagem',
                        `Você tem ${newMessages} nova(s) mensagem(ns)`,
                        'message',
                        'mensagens.php'
                    );
                    playNotificationSound();
                }

                lastMessageCount = naoLidas;
            }

            // Buscar contador de notificações
            const notifResponse = await fetch('api/notificacoes_api.php?action=contar');
            const notifData = await notifResponse.json();

            console.log('[Notificações] Resposta de notificações:', notifData);

            if (notifData.success) {
                // Suportar ambos formatos: notifData.nao_lidas ou notifData.contadores.nao_lidas
                const naoLidas = parseInt(notifData.nao_lidas || notifData.contadores?.nao_lidas) || 0;

                console.log('[Notificações] Notificações não lidas:', naoLidas);
                updateBadge('notificacoes', naoLidas);

                // Verificar se há novas notificações
                if (naoLidas > lastNotificationCount && lastNotificationCount > 0) {
                    const newNotifs = naoLidas - lastNotificationCount;
                    showToast(
                        'Nova Notificação',
                        `Você tem ${newNotifs} nova(s) notificação(ões)`,
                        'notification',
                        'notificacoes.php'
                    );
                    playNotificationSound();
                }

                lastNotificationCount = naoLidas;
            }

        } catch (error) {
            console.error('[Notificações] Erro ao atualizar contadores:', error);
        }
    }

    /**
     * Atualizar badge no menu
     */
    function updateBadge(type, count) {
        const badges = document.querySelectorAll(`.sidebar-badge[data-type="${type}"]`);

        console.log(`[Notificações] Atualizando badge ${type}:`, count, 'badges encontrados:', badges.length);

        badges.forEach(badge => {
            if (count > 0) {
                badge.textContent = count > 99 ? '99+' : count;
                badge.style.display = 'flex';
                console.log(`[Notificações] Badge ${type} EXIBIDO com valor:`, count);
            } else {
                badge.style.display = 'none';
                console.log(`[Notificações] Badge ${type} OCULTADO (count = 0)`);
            }
        });
    }

    /**
     * Enviar heartbeat para manter status online
     */
    async function sendHeartbeat() {
        try {
            const formData = new FormData();
            formData.append('action', 'heartbeat');

            await fetch('api/status_api.php', {
                method: 'POST',
                body: formData
            });

        } catch (error) {
            console.error('[Notificações] Erro ao enviar heartbeat:', error);
        }
    }

    /**
     * Mostrar toast notification
     */
    function showToast(title, message, type = 'info', link = null) {
        // Verificar se já existe container
        let container = document.getElementById('toast-container');

        if (!container) {
            container = document.createElement('div');
            container.id = 'toast-container';
            container.className = 'toast-container';
            document.body.appendChild(container);
        }

        // Criar toast
        const toast = document.createElement('div');
        toast.className = `toast toast-${type}`;

        const icon = getToastIcon(type);

        toast.innerHTML = `
            <div class="toast-icon">
                <i class="${icon}"></i>
            </div>
            <div class="toast-content">
                <div class="toast-title">${title}</div>
                <div class="toast-message">${message}</div>
            </div>
            <button class="toast-close" onclick="this.parentElement.remove()">
                <i class="fas fa-times"></i>
            </button>
        `;

        // Se houver link, tornar clicável
        if (link) {
            toast.style.cursor = 'pointer';
            toast.addEventListener('click', function(e) {
                if (!e.target.closest('.toast-close')) {
                    window.location.href = link;
                }
            });
        }

        // Adicionar ao container
        container.appendChild(toast);

        // Animar entrada
        setTimeout(() => {
            toast.classList.add('show');
        }, 10);

        // Remover automaticamente após duração
        setTimeout(() => {
            toast.classList.remove('show');
            setTimeout(() => {
                toast.remove();
            }, 300);
        }, CONFIG.TOAST_DURATION);
    }

    /**
     * Obter ícone para toast
     */
    function getToastIcon(type) {
        const icons = {
            'message': 'fas fa-envelope',
            'notification': 'fas fa-bell',
            'success': 'fas fa-check-circle',
            'error': 'fas fa-exclamation-circle',
            'warning': 'fas fa-exclamation-triangle',
            'info': 'fas fa-info-circle'
        };

        return icons[type] || icons.info;
    }

    /**
     * Tocar som de notificação
     */
    function playNotificationSound() {
        if (!CONFIG.SOUND_ENABLED) return;

        try {
            // Criar AudioContext para som simples
            const audioContext = new (window.AudioContext || window.webkitAudioContext)();
            const oscillator = audioContext.createOscillator();
            const gainNode = audioContext.createGain();

            oscillator.connect(gainNode);
            gainNode.connect(audioContext.destination);

            oscillator.frequency.value = 800;
            oscillator.type = 'sine';

            gainNode.gain.setValueAtTime(0.3, audioContext.currentTime);
            gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.5);

            oscillator.start(audioContext.currentTime);
            oscillator.stop(audioContext.currentTime + 0.5);

        } catch (error) {
            console.warn('[Notificações] Não foi possível tocar som:', error);
        }
    }

    /**
     * Iniciar polling de contadores
     */
    function startPolling() {
        if (pollingTimer) {
            clearInterval(pollingTimer);
        }

        pollingTimer = setInterval(() => {
            updateCounters();
        }, CONFIG.POLLING_INTERVAL);
    }

    /**
     * Iniciar heartbeat
     */
    function startHeartbeat() {
        if (heartbeatTimer) {
            clearInterval(heartbeatTimer);
        }

        // Enviar imediatamente
        sendHeartbeat();

        // Continuar enviando a cada intervalo
        heartbeatTimer = setInterval(() => {
            sendHeartbeat();
        }, CONFIG.HEARTBEAT_INTERVAL);
    }

    /**
     * Parar polling e heartbeat
     */
    function cleanup() {
        if (pollingTimer) {
            clearInterval(pollingTimer);
            pollingTimer = null;
        }

        if (heartbeatTimer) {
            clearInterval(heartbeatTimer);
            heartbeatTimer = null;
        }

        console.log('[Notificações] Sistema parado');
    }

    /**
     * Event listeners
     */
    function setupEventListeners() {
        // Detectar atividade do usuário para resetar heartbeat
        let activityTimeout = null;

        const resetActivity = () => {
            clearTimeout(activityTimeout);

            activityTimeout = setTimeout(() => {
                sendHeartbeat();
            }, 1000);
        };

        // Eventos de atividade
        ['mousedown', 'keydown', 'scroll', 'touchstart'].forEach(event => {
            document.addEventListener(event, resetActivity, { passive: true });
        });

        // Visibilidade da página
        document.addEventListener('visibilitychange', () => {
            if (document.hidden) {
                // Pausar polling quando página não está visível
                cleanup();
            } else {
                // Retomar quando voltar
                updateCounters();
                startPolling();
                startHeartbeat();
            }
        });
    }

    /**
     * API Pública
     */
    window.NotificacoesSystem = {
        // Atualizar contadores manualmente
        refresh: updateCounters,

        // Mostrar toast personalizado
        showToast: showToast,

        // Parar sistema
        stop: cleanup,

        // Reiniciar sistema
        restart: function() {
            cleanup();
            init();
        },

        // Configurar
        config: function(options) {
            Object.assign(CONFIG, options);
        }
    };

    // Auto-inicializar quando DOM estiver pronto
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }

})();
