<?php
// ============================================================
// GERAR QR CODE VIA LINK - baixar_qrcode.php
// ============================================================
// Este arquivo permite que o convidado baixe o QR Code
// a qualquer momento usando um link único
// ============================================================

require_once 'config.php';

// Pegar código QR da URL
$qr_code = $_GET['qr'] ?? '';

if (empty($qr_code)) {
    die('Código inválido');
}

// Buscar informações do convidado e evento
$stmt = $pdo->prepare("
    SELECT c.*, e.nome as evento_nome, e.data, e.hora, e.informacoes_importantes,
           u.nome as promoter_nome
    FROM convidados c
    INNER JOIN eventos e ON c.evento_id = e.id
    INNER JOIN users u ON c.promoter_id = u.id
    WHERE c.qr_code = ?
");
$stmt->execute([$qr_code]);
$dados = $stmt->fetch();

if (!$dados) {
    die('Código não encontrado');
}

// Função para formatar data
function formatarDataBR($data) {
    $meses = ['', 'Janeiro', 'Fevereiro', 'Março', 'Abril', 'Maio', 'Junho', 
              'Julho', 'Agosto', 'Setembro', 'Outubro', 'Novembro', 'Dezembro'];
    $partes = explode('-', $data);
    return $partes[2] . ' de ' . $meses[intval($partes[1])] . ' de ' . $partes[0];
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Download - Ingresso <?= htmlspecialchars($dados['evento_nome']) ?></title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script src="https://cdn.rawgit.com/davidshimjs/qrcodejs/gh-pages/qrcode.min.js"></script>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: Arial, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        .container {
            background: white;
            padding: 40px;
            border-radius: 20px;
            box-shadow: 0 10px 40px rgba(0,0,0,0.3);
            text-align: center;
            max-width: 500px;
        }
        h1 {
            color: #333;
            margin-bottom: 20px;
            font-size: 28px;
        }
        .info {
            background: #f5f5f5;
            padding: 20px;
            border-radius: 10px;
            margin: 20px 0;
            text-align: left;
        }
        .info p {
            margin: 10px 0;
            color: #666;
        }
        .info strong {
            color: #333;
        }
        #qrcode {
            display: inline-block;
            margin: 20px 0;
        }
        .btn {
            background: linear-gradient(135deg, #667eea, #764ba2);
            color: white;
            padding: 15px 40px;
            border: none;
            border-radius: 50px;
            font-size: 18px;
            cursor: pointer;
            margin: 10px;
            transition: transform 0.2s;
        }
        .btn:hover {
            transform: scale(1.05);
        }
        .loading {
            display: none;
            margin: 20px 0;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1><i class="fas fa-ticket-alt"></i> Seu Ingresso</h1>
        
        <div class="info">
            <p><strong>Evento:</strong> <?= htmlspecialchars($dados['evento_nome']) ?></p>
            <p><strong>Nome:</strong> <?= htmlspecialchars($dados['nome']) ?></p>
            <p><strong>Data:</strong> <?= formatarDataBR($dados['data']) ?></p>
            <p><strong>Hora:</strong> <?= date('H:i', strtotime($dados['hora'])) ?></p>
        </div>
        
        <div id="qrcode-hidden" style="display:none;"></div>

        <button class="btn" onclick="baixarIngresso()">
            <i class="fas fa-download"></i> Baixar Ingresso Completo
        </button>

        <div class="loading">
            <p><i class="fas fa-spinner fa-spin"></i> Gerando seu ingresso...</p>
        </div>
    </div>

    <script>
        // Dados do evento
        const dadosEvento = {
            nome: "<?= addslashes($dados['evento_nome']) ?>",
            data: "<?= formatarDataBR($dados['data']) ?>",
            hora: "<?= date('H:i', strtotime($dados['hora'])) ?>",
            qrcode: "<?= $qr_code ?>",
            convidado: "<?= addslashes($dados['nome']) ?>",
            promoter: "<?= addslashes($dados['promoter_nome']) ?>",
            informacoes: <?= json_encode($dados['informacoes_importantes'] ?? '') ?>
        };
        
        // Gerar QR Code oculto
        new QRCode(document.getElementById("qrcode-hidden"), {
            text: dadosEvento.qrcode,
            width: 250,
            height: 250,
            colorDark: "#000000",
            colorLight: "#ffffff",
            correctLevel: QRCode.CorrectLevel.H
        });
        
        // Aguardar QR Code carregar
        setTimeout(() => {
            document.querySelector('.loading').style.display = 'none';
        }, 1000);
        
        // FUNÇÃO: Baixar ingresso
        function baixarIngresso() {
            document.querySelector('.loading').style.display = 'block';
            
            setTimeout(() => {
                const qrCanvas = document.querySelector('#qrcode-hidden canvas');
                if (!qrCanvas) {
                    alert('Aguarde o QR Code carregar...');
                    return;
                }
                
                gerarIngresso(qrCanvas);
            }, 500);
        }
        
        function gerarIngresso(qrCanvas) {
            const canvas = document.createElement('canvas');
            const ctx = canvas.getContext('2d');
            canvas.width = 1080;
            
            // Calcular altura
            let totalHeight = 600 + 200 + 720 + 180 + 120;
            if (dadosEvento.informacoes && dadosEvento.informacoes.trim() !== '') {
                ctx.font = '28px Arial';
                const lines = wrapText(ctx, dadosEvento.informacoes, 980);
                totalHeight += 150 + (lines.length * 40) + 80;
            }
            canvas.height = totalHeight;
            
            // Fundo branco
            ctx.fillStyle = '#ffffff';
            ctx.fillRect(0, 0, canvas.width, canvas.height);
            
            let y = 0;
            
            // GRADIENTE (SEM IMAGEM)
            const gradientHeight = 600;
            const gradient = ctx.createLinearGradient(0, 0, 0, gradientHeight);
            gradient.addColorStop(0, '#667eea');
            gradient.addColorStop(1, '#764ba2');
            ctx.fillStyle = gradient;
            ctx.fillRect(0, 0, canvas.width, gradientHeight);
            
            // Nome do evento
            ctx.fillStyle = '#ffffff';
            ctx.font = 'bold 80px Arial';
            ctx.textAlign = 'center';
            const nomeEvento = dadosEvento.nome.toUpperCase();
            if (ctx.measureText(nomeEvento).width > canvas.width - 100) {
                ctx.font = 'bold 60px Arial';
            }
            ctx.fillText(nomeEvento, canvas.width/2, gradientHeight/2);
            
            y = gradientHeight + 40;
            
            // Linha
            ctx.strokeStyle = '#e0e0e0';
            ctx.lineWidth = 2;
            ctx.beginPath();
            ctx.moveTo(50, y);
            ctx.lineTo(canvas.width - 50, y);
            ctx.stroke();
            y += 50;
            
            // Nome convidado
            ctx.fillStyle = '#888888';
            ctx.font = '32px Arial';
            ctx.textAlign = 'left';
            ctx.fillText('🎫 INGRESSO DE', 50, y);
            y += 50;
            
            ctx.fillStyle = '#000000';
            ctx.font = 'bold 48px Arial';
            ctx.fillText(dadosEvento.convidado.toUpperCase(), 50, y);
            y += 60;
            
            // Linha
            ctx.strokeStyle = '#e0e0e0';
            ctx.lineWidth = 2;
            ctx.beginPath();
            ctx.moveTo(50, y);
            ctx.lineTo(canvas.width - 50, y);
            ctx.stroke();
            y += 60;
            
            // QR CODE
            const qrSize = 600;
            const qrX = (canvas.width - qrSize) / 2;
            ctx.drawImage(qrCanvas, qrX, y, qrSize, qrSize);
            y += qrSize + 30;
            
            ctx.fillStyle = '#666666';
            ctx.font = '28px Courier New';
            ctx.textAlign = 'center';
            ctx.fillText(dadosEvento.qrcode, canvas.width/2, y);
            y += 60;
            
            // Linha
            ctx.strokeStyle = '#e0e0e0';
            ctx.lineWidth = 2;
            ctx.beginPath();
            ctx.moveTo(50, y);
            ctx.lineTo(canvas.width - 50, y);
            ctx.stroke();
            y += 60;
            
            // Data e hora
            ctx.fillStyle = '#888888';
            ctx.font = '28px Arial';
            ctx.textAlign = 'left';
            ctx.fillText('📅 Data', 50, y);
            ctx.fillStyle = '#000000';
            ctx.font = 'bold 36px Arial';
            ctx.fillText(dadosEvento.data, 50, y + 45);
            
            ctx.fillStyle = '#888888';
            ctx.font = '28px Arial';
            ctx.textAlign = 'right';
            ctx.fillText('🕒 Horário', canvas.width - 50, y);
            ctx.fillStyle = '#000000';
            ctx.font = 'bold 36px Arial';
            ctx.fillText(dadosEvento.hora, canvas.width - 50, y + 45);
            y += 120;
            
            // Informações importantes
            if (dadosEvento.informacoes && dadosEvento.informacoes.trim() !== '') {
                ctx.strokeStyle = '#e0e0e0';
                ctx.lineWidth = 2;
                ctx.beginPath();
                ctx.moveTo(50, y);
                ctx.lineTo(canvas.width - 50, y);
                ctx.stroke();
                y += 50;
                
                ctx.fillStyle = '#FF6B6B';
                ctx.font = 'bold 36px Arial';
                ctx.textAlign = 'left';
                ctx.fillText('⚠️ INFORMAÇÕES IMPORTANTES', 50, y);
                y += 50;
                
                const infoLines = wrapText(ctx, dadosEvento.informacoes, 980);
                const infoBoxHeight = (infoLines.length * 40) + 60;
                
                ctx.fillStyle = '#f8f8f8';
                ctx.fillRect(50, y, canvas.width - 100, infoBoxHeight);
                ctx.strokeStyle = '#FF6B6B';
                ctx.lineWidth = 3;
                ctx.strokeRect(50, y, canvas.width - 100, infoBoxHeight);
                y += 40;
                
                ctx.fillStyle = '#000000';
                ctx.font = '28px Arial';
                infoLines.forEach(line => {
                    ctx.fillText('• ' + line, 80, y);
                    y += 40;
                });
                y += 40;
            }
            
            // Rodapé
            ctx.strokeStyle = '#e0e0e0';
            ctx.lineWidth = 2;
            ctx.beginPath();
            ctx.moveTo(50, y);
            ctx.lineTo(canvas.width - 50, y);
            ctx.stroke();
            y += 40;
            
            ctx.fillStyle = '#888888';
            ctx.font = '24px Arial';
            ctx.textAlign = 'center';
            ctx.fillText('Convidado por: ' + dadosEvento.promoter, canvas.width/2, y);
            y += 35;
            ctx.fillText('PromoSys desenvolvido por Alerson Araújo • Licenciado para Espaço Taal', canvas.width/2, y);
            
            // Download
            canvas.toBlob(function(blob) {
                const url = URL.createObjectURL(blob);
                const link = document.createElement('a');
                const nomeArquivo = `Ingresso_${dadosEvento.nome.replace(/[^a-z0-9]/gi, '_')}_${dadosEvento.convidado.replace(/[^a-z0-9]/gi, '_')}.png`;
                link.download = nomeArquivo;
                link.href = url;
                link.click();
                URL.revokeObjectURL(url);
                
                document.querySelector('.loading').style.display = 'none';
                alert('Ingresso baixado com sucesso!');
            }, 'image/png', 1.0);
        }
        
        function wrapText(context, text, maxWidth) {
            const words = text.split(' ');
            const lines = [];
            let currentLine = words[0];
            for (let i = 1; i < words.length; i++) {
                const word = words[i];
                const width = context.measureText(currentLine + ' ' + word).width;
                if (width < maxWidth) {
                    currentLine += ' ' + word;
                } else {
                    lines.push(currentLine);
                    currentLine = word;
                }
            }
            lines.push(currentLine);
            return lines;
        }
    </script>
    </div><!-- /.main-content -->
    <script src="assets/js/sidebar.js"></script>
</body>
</html>
